#----------------------------------------------------------------
# Toy model unfolding: simulation functions
# 
# These functions are used to simulate toy model 'MC' and 'data'
# events to be unfolded.
#
# Obviously, the user has to replace them by what describes
# at best the physics case under study
#
#   fun(x, IFLAG)
#   rcat(n, x, p, binned=TRUE)
#   set.up.pec(IFLAG )
#   extract(nevts, nfun) 
#   apply.smearing(evts, nsmear, MC=FALSE)
#
#  G. D'Agostini, December 2005
#----------------------------------------------------------------

# dimensionality of the problem
nC <- 10
nE <- 14

# fun(x, IFLAG) -------------------------------------------------
#
# 'true' functions
# defined between 0 e 10

fun <- function(x, IFLAG) {
   if ( IFLAG == 1 ) {
     f = 11 - x
   } else if (IFLAG == 2) {
     f = x^2
   } else if (IFLAG == 3) {
     f = (x-5)^2
   } else if (IFLAG == 4) {
     f = 40 - (x-5)^2
   } else if (IFLAG == 5) {
     f = x
     f[ x <= 5] = 10 - f[ x <= 5]
   } else if (IFLAG == 6) {
     f = 40 - abs(x-5)^1.5 * sign(x-5)
   } else {
     f =  rep(1, length(x))  # default uniform
   }
   f[ x < 0] = 0
   f[ x > 10] = 0
   return (f)
 }


# rcat(n, x, p, binned=TRUE)  ------------------------------------------------
#
#- categorical random generator (a la BUGS)
#- (It could have been done in other ways. Anyhow, it works
#   and it is no time-critical)
#
# IN  : n     : number of random numbers
#       p     : a vector of probability values, i.e. sum(p) = 1
#               the n numbers are generated according to p
#       x     : a vector of the same length of p: they are the values corresponding
#               to the probabilities p [i.e. P(x[1]) = p[1], etc.].
#               Only needed if unbinned numbers are required
#       binned: choose to have binned numbers (for this unfolding this is
#               what is needed; the opzion is just to have a more general function
#               for further applications)
# OUT : a vector of length length(p) containing the number of 'events' per bin
#               ("binned=TRUE") or a vector of length n containing the individual
#               'events'

rcat <- function(n, x, p, binned=TRUE) {
  # check normalization
  if (abs(sum(p)-1) > 1.0e-10)            return(0)
  # x is indeed needed only if binned=FALSE
  if (!binned && (length(x) != length(p))) return(0)

  r <- as.vector(rmultinom(1, n, p))

  if (!binned) {
     res = numeric()
     # ordered list
     for ( i in 1:length(p) ){
       res = c(res, rep(x[i],r[i]) )
     }
     # random permutation
     res <- sample(res)
     return(res)         # random events are returned
  } else {
    return(r)            # numbers of events per bin are returned
  }
}

# set.up.pec( IFLAG ) ----------------------------------------------
# 
#- smearing functions (as NIM paper)
#
#  IN: IFLAG: it can be 1 or 2 to choose between two possibilities
# OUT: array pec, i.e. P(Ej|Ci), of size nE*nC

set.up.pec <- function(IFLAG ) {
   pec <- matrix (rep(0, nC*nE), nE, nC)
   if ( IFLAG == 0) {
     diag(pec) <- rep(100, nC)
   } else if (IFLAG == 1) {
     pec[1:4, 1]   <- c(10, 25, 40, 25) * 0.1
     pec[4:6, 2]   <- c(25, 50, 25)     * 0.2
     pec[6:8, 3]   <- c(40, 40, 20)     * 0.3
     pec[8:10, 4]   <- c(40, 40, 20)    * 0.4
     pec[8:11, 5]  <- c(5, 20, 50, 25)  * 0.5
     pec[9:11, 6]  <- c(20, 60, 20)     * 0.6
     pec[9:11, 7]  <- c(20, 40, 40)     * 0.7
     pec[10:12, 8] <- c(25, 50, 25)     * 0.8
     pec[11:13, 9] <- c(25, 50, 25)     * 0.9
     pec[12:14, 10]<- c(25, 50, 25)     * 1.0
   } else if (IFLAG == 2) {
     pec[14:11, 1] <- c(10, 20, 30, 20)
     pec[3:1, 1]   <- c(5, 10, 5)
     pec[11:8, 2]  <- c(20, 40, 15, 5)
     pec[4:2, 2]   <- c(6, 8, 6)
     pec[14:13, 3] <- c(5, 5)
     pec[10:7, 3]  <- c(5, 30, 30, 15)
     pec[4:3,  3]  <- c(5, 5)
     pec[12, 4]    <- 5
     pec[8:4, 4]     <- c(5, 30, 35, 20, 5)
     pec[10:9, 5]  <- c(5, 5)
     pec[7:4, 5]   <- c(5, 20, 40, 25)
     pec[10:9, 6]  <- c(5, 5)
     pec[6:4, 6]   <- c(20, 50, 20)
     pec[8, 7]     <- 5
     pec[6:4, 7]   <- c(20, 35, 40)
     pec[7, 8]     <- 5
     pec[5:3, 8]   <- c(25, 45, 25)
     pec[8, 9]     <- 5
     pec[6, 9]     <- 5
     pec[4:2, 9]   <- c(25, 40, 25)
     pec[9, 10]    <- 5
     pec[3:1, 10]  <- c(25, 45, 25)
     pec = pec * 0.9
   }
   pec <- pec/100.
   return(pec)
}

# extract(nevts, nfun) ---------------------------------------------
#
# event generation
#
#  IN: nevts: number of events to generate
#    : nfun:  a number to choose among several possibilities (0, 1, ..., 6)
# OUT: vector of length nC containing the number of events per bin

extract <- function (nevts, nfun) {
   low <- seq(0, nC-1)
   up <- low +1
   integrals <- numeric()
   for (i in 1:length(low)) {
     integrals[i] = as.numeric(integrate(fun, low[i], up[i], IFLAG=nfun)[1])
   }
   probs =  integrals/sum(integrals)
   evts <- rcat(nevts, 0, probs, binned=TRUE)
   return(evts)
}

# apply.smearing(evts, nsmear, MC=FALSE) --------------------------------------
#
# smearing
#
#  IN:   evts: vector of length nC containing the number of events per bin
#      nsmear: kind of smearing (1 or 2 available in this demo)
#          MC: if TRUE the function provides the number of events in each
#              effect-bin for each cause-bin;
#              if FALSE the function only provides the total number of events
#              in the effects bins
# OUT: array of dimensions nE*nC, or vector of length nE, depending on MC

apply.smearing <- function (evts, nsmear, MC=FALSE) {
   pec <- set.up.pec(nsmear)
   obs <- rep(0, nE)
   obsi <- matrix(rep(0, nC*nE), nE, nC)
   for (i in 1: nC) {
      probs <- pec[,i]                   # smearing probabilities
      probs <- c(probs, 1 - sum(probs))  # add inefficiency
      obsi[,i] <- rcat(evts[i], 0, probs, binned=TRUE)[-(nE+1)] # remove last bin
      obs <- obs + obsi[,i]
    }
   if (MC) {
     return(obsi)  # details to evaluate smearing matrix
   } else {
     return(obs)   # only observables
   }
}
